/**
 * Handles the supported page formats for printings.
 */
class UIPrintFormat {
	/**
	 * Class constructor.
	 *
	 * @param 	string 	format  The page format.
	 */
	constructor(format, size) {
		// keep the given format
		this.format = format;
		
		// format page size
		this.size = {
			width: 0,
			height: 0,
		};

		// format print size
		this.printSize = null;

		// value to use for CSS @page
		this.ISO = format;

		// fetch size
		switch (this.format) {
			case 'A3':
				this.size.width  = 297;
				this.size.height = 420;
				break;

			case 'A4':
				this.size.width  = 210;
				this.size.height = 297;
				break;

			case 'A5':
				this.size.width  = 148;
				this.size.height = 210;
				break;

			case 'Letter':
				this.size.width  = 216;
				this.size.height = 279;
				break;

			case 'Legal':
				this.size.width  = 216;
				this.size.height = 356;
				break;

			case 'Tabloid':
				this.size.width  = 279;
				this.size.height = 432;
				break;

			case 'custom':
				this.size.width  = size.width;
				this.size.height = size.height;

				this.printSize = {
					width:  size.printWidth,
					height: size.printHeight,
				};
				break;
		}
	}

	/**
	 * Returns the page size.
	 *
	 * @return 	Size
	 */
	getSize() {
		return this.size;
	}

	/**
	 * Returns the page width.
	 *
	 * @return 	float
	 */
	getWidth() {
		return this.size.width;
	}

	/**
	 * Returns the page height.
	 *
	 * @return 	float
	 */
	getHeight() {
		return this.size.height;
	}

	/**
	 * Returns the size of the print format.
	 *
	 * @return 	Size
	 */
	getPrintSize() {
		return this.printSize || this.getSize();
	}

	/**
	 * Returns the width of the print format.
	 *
	 * @return 	float
	 */
	getPrintWidth() {
		return this.printSize ? this.printSize.width : this.getWidth();
	}

	/**
	 * Returns the height of the print format.
	 *
	 * @return 	float
	 */
	getPrintHeight() {
		return this.printSize ? this.printSize.height : this.getHeight();
	}

	/**
	 * Returns the size CSS useful for @page selector.
	 *
	 * @return 	string
	 */
	getPageSizeCSS() {
		let size;

		if (this.ISO !== 'custom') {
			// use specified page format
			size = this.ISO;
		} else {
			// use given size
			size = this.getWidth() + 'mm ' + this.getHeight() + 'mm';
		}

		// create CSS rule
		return 'size: ' + size + ';';
	}
}

// export for external usage
module.exports = UIPrintFormat;
